<?php
/**
 * MinervaKB Elementor Related Articles Widget
 * Copyright: 2015-2020 @KonstruktStudio
 */

use \Elementor\Widget_Base;
use \Elementor\Controls_Manager;
use \Elementor\Repeater;

class MinervaKB_ElementorRelatedWidget extends Widget_Base {

    public function get_name() {
        return 'minervakb-related';
    }

    public function get_title() {
        return __( 'KB Related Articles', 'minerva-kb' );
    }

    public function get_icon() {
        return 'eicon-time-line';
    }

    public function get_categories() {
        return array('minerva-support');
    }

    protected function _register_controls() {

        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'minerva-kb'),
                'tab' => Controls_Manager::TAB_CONTENT,
            )
        );

        $this->add_control(
            'title',
            [
                'label' =>  __( 'Title (optional)', 'minerva-kb' ),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'default' => '',
                'placeholder' => __('Type your title here', 'minerva-kb'),
                'description' => __( 'By default, the title from global settings is used', 'minerva-kb' ),
            ]
        );

        $this->add_control(
            'mode',
            [
                'label' => __( 'Related articles mode', 'minerva-kb' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'manual',
                'label_block' => true,
                'options' => [
                    'manual' => __( 'Manual', 'minerva-kb' ),
                    'topic' => __( 'Current topic', 'minerva-kb' ),
                ],
                'description' => __( 'Choose articles manually or display articles from current topic', 'minerva-kb' )
            ]
        );

        $this->add_control(
            'limit',
            [
                'label' => __( 'Limit articles from current topic', 'minerva-kb' ),
                'type' => Controls_Manager::NUMBER,
                'min' => -1,
                'max' => 99,
                'step' => 1,
                'default' => 5,
                'condition' => [
                    'mode' => 'topic',
                ],
                'description' => __( 'For Current topic mode. Use -1 to show all.', 'minerva-kb' )
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'article_id',
            [
                'label' => __('Search & Select KB Article', 'minerva-kb'),
                'type' => Controls_Manager::SELECT2,
                'options' => $this->get_all_articles(),
                'label_block' => true,
                'multiple' => false
            ]
        );

        $this->add_control(
            'articles',
            [
                'label' => __( 'Related Articles List', 'minerva-kb' ),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [],
                'title_field' => '{{{ MinervaKBElementor && MinervaKBElementor.articles && MinervaKBElementor.articles[article_id] && MinervaKBElementor.articles[article_id].title }}}',
            ]
        );

        $this->end_controls_section();
    }

    private function get_all_articles() {
        $posts = get_posts([
            'post_type' => MKB_Options::option('article_cpt'),
            'post_status' => 'publish',
            'posts_per_page' => '-1'
        ]);

        if (!empty($posts)) {
            return wp_list_pluck($posts, 'post_title', 'ID');
        }

        return [];
    }

    /**
     * TODO: move all related articles renders to template helper
     */
    protected function render() {
        $settings = $this->get_settings_for_display();

        $title = isset($settings["title"]) && $settings["title"] ? $settings["title"] : MKB_Options::option('related_content_label');
        $mode = isset($settings['mode']) ? $settings['mode'] : 'manual';
        $limit = isset($settings['limit']) ? $settings['limit'] : 5;
        $ids = array();

        foreach($settings['articles'] as $article) {
            if (isset($article['article_id']) && $article['article_id']) {
                array_push($ids, (int)$article['article_id']);
            }
        }

        if ($mode === 'topic') {
            $ids = array();

            if (!is_singular(MKB_Options::option('article_cpt'))) {
                echo __('<p>Warning! Current topic mode works only for KB articles</p>', 'minerva-kb');

                return;
            }

            $term = MKB_TemplateHelper::get_article_main_topic(get_the_ID());

            if (!$term) {
                return;
            }

            $query_args = array(
                'post_type' => MKB_Options::option( 'article_cpt' ),
                'ignore_sticky_posts' => 1,
                'posts_per_page' => $limit,
                'post__not_in' => array(get_the_ID()),
                'tax_query' => array(
                    array(
                        'taxonomy' => MKB_Options::option('article_cpt_category'),
                        'field' => 'slug',
                        'terms' => $term->slug,
                    ),
                )
            );

            $topic_loop = new WP_Query( $query_args );

            if ($topic_loop->have_posts()):
                while ( $topic_loop->have_posts() ) : $topic_loop->the_post();
                    array_push($ids, get_the_ID());
                endwhile;
            endif;

            wp_reset_postdata();
        }

        if ($ids && is_array($ids) && !empty($ids)):
            ?>
            <div class="mkb-related-content">
                <div class="mkb-related-content-title"><?php echo esc_html($title); ?></div>
                <ul class="mkb-related-content-list">
                    <?php foreach($ids as $index => $id):
                        if ( empty($id) || !is_string( get_post_status( $id )) ) {
                            continue;
                        }
                        ?>
                        <li><a href="<?php echo esc_url(get_permalink($id)); ?>"><?php echo esc_html(get_the_title($id)); ?></a></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php
        endif;
    }
}